<?php

namespace Cpay\Woocommerce;

use WC_Payment_Gateway;

final class Gateway extends WC_Payment_Gateway
{
    /**
     * @var string
     */
    public $id = 'cpay';
    /**
     * @var bool
     */
    public $has_fields = true;
    /**
     * @var string
     */
    public $title = 'Pay with Cpay';
    /**
     * @var string
     */
    public $method_title = 'Cpay';
    /**
     * @var string
     */
    public $method_description = "";
    /**
     * @var string
     */
    public $checkout_url;
    /**
     * @var string
     */
    private $logo_theme;

    public function __construct()
    {
        $this->title = $this->get_option('method_title') ?: $this->title;
        $this->method_description = "
        <img src='https://assets-global.website-files.com/64ea031997a309c33139d1c1/6528324ae421ea9f7879a059_footer.svg' style='max-width: 500px; height: auto;'>
        <p>To start using Cpay, you need to follow a few simple steps.</p>

        <p>1) Create an account on our website <a href='https://app.cpay.world/sign-up?lang=en'>cpay.world</a> and complete the verification process.</p>
        <p>2) Go to the checkouts section, create a checkout with the cart type. There you can select the currencies you will receive for paying for the goods.</p>
        <p>3) Copy the checkout link and paste it into the plugin settings.</p>
        <p>4) Set callback url in merchant settings - https://your-site-url.com/wp-json/cpay-webhook/v1</p>
        <p>Congratulations! Now you can accept cryptocurrency payments using Cpay.</p>
        <p>Have questions? We are always happy to answer on <a href='https://cpay.world/'>CPAY</a>. Our support operates 24 hours a day.</p>
        ";
        $this->description = $this->get_option('description');
        $this->form_fields = $this->adminFields();
        $this->init_settings();

        $this->logo_theme = $this->get_option('logo_theme') ?: 'light';

        $path = str_replace(ABSPATH, '', __DIR__) . "/images/logo_$this->logo_theme.svg";
        $this->icon = esc_url(get_option('cpay_method_image')) ?: site_url($path);

        add_action("woocommerce_update_options_payment_gateways_{$this->id}", array($this, 'process_admin_options'));
    }

    /**
     * @return array
     */
    public function adminFields()
    {
        return [
            'enabled' => [
                'title' => __('Enabled'),
                'type' => 'checkbox',
                'default' => 'no',
                'desc_tip' => true
            ],
            'checkout_url' => [
                'title' => '
                Checkout url 
                <p><font size="1">You can find the Checkout url in the checkouts of your personal account. </p>
                ',
                'type' => 'text'
            ],
            'method_title' => [
                'title' => 'Method title
                <p><font size="1">Payment method name. For example "Cpay".</font></p>
                ',
                'type' => 'text',
                'default' => 'Pay with Cpay'
            ],
            'description' => [
                'title' => 'Method description
                <p><font size="1">
                Description that will be located next to the Cpay payment method. For example "Pay with cryptocurrency." </font></p>
                ',
                'type' => 'text',
                'default' => 'Crypto payment system'
            ],
            'method_image' => [
                'title' => 'Method Image</p>
                ',
                'type' => 'file',
                'desc_tip' => true,
                'description' => 'Upload an image for the payment method',
            ],
        ];
    }

    /**
     * @param $order_id
     * @return array
     */
    public function process_payment($order_id)
    {
        $order = wc_get_order($order_id);
        $order->update_status(PaymentStatus::WC_STATUS_PENDING);
        $order->save();

        wc_reduce_stock_levels($order_id);
        WC()->cart->empty_cart();

        $checkout_url = sprintf('%s?clickId=%s&amount=%s&fiatCurrency=%s&returnUrl=%s', $this->settings['checkout_url'], (string) $order_id, (string) $order->get_total(), (string) $order->get_currency(), get_site_url());

        return ['result' => 'success', 'redirect' => $checkout_url];
    }

    public function process_admin_options()
    {
        parent::process_admin_options();

        $uploaded_image = isset($_FILES['woocommerce_cpay_method_image']) ? $_FILES['woocommerce_cpay_method_image'] : null;

        if ($uploaded_image && isset($uploaded_image['tmp_name']) && !empty($uploaded_image['tmp_name'])) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            $upload_overrides = array('test_form' => false);
            $movefile = wp_handle_upload($uploaded_image, $upload_overrides);

            if ($movefile && !isset($movefile['error'])) {
                $image_url = $movefile['url'];
                update_option('cpay_method_image', $image_url);
            }
        }
    }

    public function admin_options()
    {
        $image_url = get_option('cpay_method_image'); // Replace with your option name

        echo '<h2>' . esc_html($this->method_title) . '</h2>';
        echo '<div>' . $this->method_description . '</div>';

        if (!empty($image_url)) {
            echo '<h3>' . __('Image Preview', 'woocommerce') . '</h3>';
            echo '<img src="' . esc_url($image_url) . '" alt="Method Image" style="max-width: 200px; height: auto;" /><br />';
        }

        echo '<table class="form-table">';
        // Render other settings fields here...
        $this->generate_settings_html();
        echo '</table>';
    }

    public function is_available()
    {
        if ($this->get_option('enabled') != 'yes') {
            return false;
        }

        if (empty($this->get_option('checkout_url'))) {
            return false;
        }

        return true;
    }
}
