<?php

/**
 * Plugin Name: Cpay
 * Plugin URI: https://docs.cpay.world/
 * Description: CPAY is a leading software development company specializing in crypto payment software products and services. Our goal is to provide a software solution for building a complete crypto payment infrastructure. Our offerings range from Crypto Payment Gateways and Crypto Wallets to comprehensive solutions for AML (Anti-Money Laundering) and KYC (Know Your Customer) compliance. CPAY aims to make the integration of crypto transactions as seamless as possible across any platform or business.
 * Version: 1.0.0
 * Author: cpay.world
 * Author URI: https://cpay.world/
 * Developer: Cpay
 * Developer https://cpay.world/
 *
 * @package WooCommerce\Admin
 */

use Cpay\Api\Modules\Webhook;
use Cpay\Woocommerce\PaymentStatus;
use Cpay\Woocommerce\BlockGateway;

defined('ABSPATH') || exit;

require_once __DIR__ . '/vendor/autoload.php';

if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
    return;
}

add_filter('woocommerce_payment_gateways', function ($plugins) {
    return array_merge([\Cpay\Woocommerce\Gateway::class], $plugins);
});

add_filter('plugin_action_links_' . plugin_basename(__FILE__), function ($links) {
    $url = admin_url('admin.php?page=wc-settings&tab=checkout&section=cpay');
    return array_merge(['<a href="' . $url . '">' . __('Configure') . '</a>'], $links);
});

add_action('plugins_loaded', function () {
    return new \Cpay\Woocommerce\Gateway();
});

add_filter('wc_order_statuses', function ($statuses) {
    $statuses['wc-wrong-amount'] = __('Wrong amount');
    return $statuses;
});

// Register block-based payment methodAdd commentMore actions
add_action('woocommerce_blocks_loaded', function () {
    if (class_exists('Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType')) {
        add_action('woocommerce_blocks_payment_method_type_registration', function ($payment_method_registry) {
            $payment_method_registry->register(new BlockGateway());
        });
    }
});

// Add payment method data to blocks
add_filter('woocommerce_blocks_payment_method_data', function ($payment_method_data, $payment_method) {
    if ($payment_method instanceof BlockGateway) {
        $gateway = new \Cpay\Woocommerce\Gateway();
        $payment_method_data['cpay_data'] = [
            'title' => $gateway->title,
            'description' => $gateway->description,
            'supports' => ['products', 'refunds'],
        ];
    }
    return $payment_method_data;
}, 10, 2);

add_action('rest_api_init', function () {
    register_rest_route('cpay-webhook', 'v1', array(
        'methods' => 'POST',
        'permission_callback' => function () {
            return true;
        },
        'callback' => function ($request) {
            try {
                $header = $request->get_header('X-Authorization');
                $body = $request->get_json_params();
                if (!$header || !$body) {
                    return ['success' => false];
                }
                $webhookResult = Webhook::handleWebhook($header, $body);
                if ($webhookResult) {
                    if (isset($webhookResult['outsideOrderId']) && $webhookResult['typeTransaction'] == 'Replenishment' && $webhookResult['systemStatus'] == 'Done') {
                        $orderId = $webhookResult['outsideOrderId'];

                        $order = wc_get_order($orderId);

                        $order->set_status($webhookResult['chargeStatus'] != 'Done' ? PaymentStatus::WC_STATUS_WRONG_AMOUNT : PaymentStatus::convertToWoocommerceStatus($webhookResult['systemStatus']));
                        $order->save();

                        if (PaymentStatus::isNeedReturnStocks($webhookResult['systemStatus'])) {
                            wc_increase_stock_levels($order);
                        }

                        return ['success' => true];
                    }

                    return ['success' => false];
                }
            } catch (Exception $e) {
                return ['success' => false];
            }

            return ['success' => true];
        }
    ));
});
